"""
Z240-MP4をエンドデバイス(温度センサー)として動作させる

実行方法:
    python run.py COM1
"""

import serial
import struct
import sys
import time


# シリアルポートの設定
# コマンドライン引数でポート名(例:COM1,/dev/ttyUSB0)を指定する
PORT = sys.argv[1]
BAUDRATE = 115200
TIMEOUT = 1.0
READ_SIZE = 255

# スリープ型エンドデバイスとする場合はTrueを指定する
IS_SLEEPY_DEVICE = False


def write_data_to_module(ser: serial.Serial, data: bytes) -> None:
    # スリープ型エンドデバイスの場合、ウェイクアップのために先頭に0x00を6バイト書き込む必要がある
    if IS_SLEEPY_DEVICE:
        ser.write(bytes([0x00, 0x00, 0x00, 0x00, 0x00, 0x00]) + data)
    else:
        ser.write(data)
    print("WRITE:", ' '.join('%02X' % b for b in data))


def read_data_from_module(ser: serial.Serial) -> bytes:
    ret = ser.read(size=READ_SIZE)
    if len(ret) > 0:
        print("READ: ", ' '.join('%02X' % b for b in ret))
    return ret


def calc_xor8(data: bytes) -> int:
    # チェックサムを計算
    xor = 0
    for d in data:
        xor = xor ^ d

    return xor


def write_attribute(ser: serial.Serial, port_idx: int, direction: int, cluster_id: int,
                    manufacturer_code: int, attr_id: int, attr_value: int) -> None:
    # 属性値を設定する(コマンドコード:0x43)

    # コマンドデータを作成
    cmd_data = bytearray([0x00, 0x43, 0x01, port_idx, direction])
    cmd_data += struct.pack('<H', cluster_id)
    cmd_data += struct.pack('<H', manufacturer_code)
    cmd_data += struct.pack('<H', attr_id)
    cmd_data += struct.pack('<h', attr_value)
    checksum = calc_xor8(cmd_data)
    cmd_data += bytes([checksum])
    frame_data = bytes([0x55, len(cmd_data)]) + cmd_data

   # コマンドを実行
    write_data_to_module(ser, frame_data)
    read_data_from_module(ser)


def update_sensor_data(ser: serial.Serial, temperature: float) -> None:

    print("temperature: %.2f" % temperature)

    port_idx = 0
    direction = 0
    manufacturer_code = 0x0000

    # クラスタIDの指定 Temperature Measurement (0x0402)
    cluster_id = 0x0402
    # 属性IDの指定 MeasuredValue (0x0000)
    attr_id = 0x0000
    # 温度の値は 0.01度単位の整数にして書き込む
    attr_value = int(temperature * 100)
    write_attribute(ser, port_idx, direction, cluster_id, manufacturer_code, attr_id, attr_value)


# シリアルポートを開く
ser = serial.Serial(port=PORT, baudrate=BAUDRATE, timeout=TIMEOUT)

print("Starting the temperature sensor device.")

# センサー値をコーディネータに送信する間隔を300秒に設定
interval = 3
next_time = time.time() + interval

# センサー値のダミーデータ
dummy_temperature = [5, 10, 15, 20, 25, 30, 35, 40]

idx = 0

while True:
    # 送信予定時刻を過ぎていれば温度の属性値を更新する
    current_time = time.time()
    if current_time >= next_time:
        update_sensor_data(ser, dummy_temperature[idx])
        idx = (idx + 1) % 8
        next_time += interval

    time.sleep(0.1)
