"""
Z240-MP4を温度センサーとしてデバイス設定し、コーディネータ(ZigBee Hub)に接続する

実行方法:
    python config.py COM1
"""

import serial
import struct
import sys
import time


# シリアルポートの設定
# コマンドライン引数でポート名(例:COM1,/dev/ttyUSB0)を指定する
PORT = sys.argv[1]
BAUDRATE = 115200
TIMEOUT = 0.2
READ_SIZE = 255

# スリープ型エンドデバイスとする場合はTrueを指定する
IS_SLEEPY_DEVICE = False


def write_data_to_module(ser: serial.Serial, data: bytes) -> None:
    # スリープ型エンドデバイスの場合、ウェイクアップのために先頭に0x00を6バイト書き込む必要がある
    if IS_SLEEPY_DEVICE:
        ser.write(bytes([0x00, 0x00, 0x00, 0x00, 0x00, 0x00]) + data)
    else:
        ser.write(data)
    print("WRITE:", ' '.join('%02X' % b for b in data))


def read_data_from_module(ser: serial.Serial) -> bytes:
    ret = ser.read(size=READ_SIZE)
    print("READ: ", ' '.join('%02X' % b for b in ret))
    return ret


def calc_xor8(data: bytes) -> int:
    # チェックサムを計算
    xor = 0
    for d in data:
        xor = xor ^ d

    return xor


def add_attribute_to_zcl_port(ser: serial.Serial, cluster_id: int,  attr: dict) -> None:
    # ポートに属性を追加する(コマンドコード:0x41)

    # 製造者コードは0x0000を設定する
    manufacturer_code = 0x0000

    # コマンドデータを作成
    cmd_data = bytearray([0x00, 0x41])
    cmd_data += struct.pack('<H', cluster_id)
    cmd_data += struct.pack('<H', manufacturer_code)
    cmd_data += struct.pack('<H', attr.get('attr_id'))
    cmd_data += bytes([attr.get('data_type')])
    cmd_data += bytes([attr.get('operation')])
    cmd_data += bytes([attr.get('direction')])
    checksum = calc_xor8(cmd_data)
    cmd_data += bytes([checksum])
    frame_data = bytes([0x55, len(cmd_data)]) + cmd_data

    # コマンドを実行
    write_data_to_module(ser, frame_data)
    read_data_from_module(ser)


def set_zcl_port_configuration(ser: serial.Serial) -> None:
    # モジュールを温度センサーデバイスとして設定するサンプル

    # 作成するポート番号を1にセット
    port_number = 1

    # Zigbee Cluster Libraryから温度センサーに対応するプロファイルID、デバイスIDをセット
    profile_id = 0x0104  # Home Automation
    device_id = 0x0302  # Temperature Sensor

    # 入力クラスタおよび出力クラスタのリストを作成
    input_clusters = [0x0000, 0x0003, 0x0402, 0x0405]
    # Cluster ID
    # Basic (0x0000)
    # Identify (0x0003)
    # Temperature Measurement (0x0402)
    # Relative Humidity Measurement (0x0405)

    # 出力クラスタは無し
    output_clusters = []

    # 各クラスタに設定する属性のリストを作成
    # Zigbee Cluster LibraryからAttribute Identifier,データ型等を設定
    basic_attrs = [
        {'attr_id': 0x0000, 'data_type': 0x20, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0x0001, 'data_type': 0x20, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0x0002, 'data_type': 0x20, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0x0003, 'data_type': 0x20, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0x0007, 'data_type': 0x30, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0xFFFD, 'data_type': 0x21, 'operation': 0x01, 'direction': 0x00}
    ]
    identify_attrs = [
        {'attr_id': 0x0000, 'data_type': 0x21, 'operation': 0x03, 'direction': 0x00},
        {'attr_id': 0xFFFD, 'data_type': 0x21, 'operation': 0x01, 'direction': 0x00}
    ]
    temperature_attrs = [
        {'attr_id': 0x0000, 'data_type': 0x29, 'operation': 0x05, 'direction': 0x00},
        {'attr_id': 0x0001, 'data_type': 0x29, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0x0002, 'data_type': 0x29, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0xFFFD, 'data_type': 0x21, 'operation': 0x01, 'direction': 0x00}
    ]
    humidity_attrs = [
        {'attr_id': 0x0000, 'data_type': 0x21, 'operation': 0x05, 'direction': 0x00},
        {'attr_id': 0x0001, 'data_type': 0x21, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0x0002, 'data_type': 0x21, 'operation': 0x01, 'direction': 0x00},
        {'attr_id': 0xFFFD, 'data_type': 0x21, 'operation': 0x01, 'direction': 0x00}
    ]

    # ポートに追加する属性の総数をカウント
    all_attrs_count = len(basic_attrs) + len(identify_attrs) + \
        len(temperature_attrs) + len(humidity_attrs)

    # ポートの作成(コマンドコード:0x40)
    print("Create ZCL port.")

    # コマンドデータを作成
    cmd_data = bytearray([0x00, 0x40])
    cmd_data += bytes([all_attrs_count])
    cmd_data += bytes([port_number])
    cmd_data += struct.pack('<H', profile_id)
    cmd_data += struct.pack('<H', device_id)
    cmd_data += bytes([len(input_clusters)])
    for cluster_id in input_clusters:
        cmd_data += struct.pack('<H', cluster_id)

    cmd_data += bytes([len(output_clusters)])
    for cluster_id in output_clusters:
        cmd_data += struct.pack('<H', cluster_id)

    checksum = calc_xor8(cmd_data)
    cmd_data += bytes([checksum])
    frame_data = bytes([0x55, len(cmd_data)]) + cmd_data

    # コマンドを実行
    write_data_to_module(ser, frame_data)
    read_data_from_module(ser)

    # ポートに属性を追加する(コマンドコード:0x41)
    print("Add attribute to ZCL port.")
    for attr in basic_attrs:
        add_attribute_to_zcl_port(ser, input_clusters[0], attr)

    for attr in identify_attrs:
        add_attribute_to_zcl_port(ser, input_clusters[1], attr)

    for attr in temperature_attrs:
        add_attribute_to_zcl_port(ser, input_clusters[2], attr)

    for attr in humidity_attrs:
        add_attribute_to_zcl_port(ser, input_clusters[3], attr)

    # ポート設定を保存する(コマンドコード:0x42)
    write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x42, 0x42]))
    read_data_from_module(ser)

# シリアルポートを開く
ser = serial.Serial(port=PORT, baudrate=BAUDRATE, timeout=TIMEOUT)

# モジュールをエンドデバイスに設定(コマンドコード:0x05)
print("set device type to End Device")
if IS_SLEEPY_DEVICE:
    write_data_to_module(ser, bytes([0x55, 0x04, 0x00, 0x05, 0x03, 0x06]))
else:
    write_data_to_module(ser, bytes([0x55, 0x04, 0x00, 0x05, 0x02, 0x07]))
read_data_from_module(ser)

# 現在のポート設定をクリアする(コマンドコード:0x44)
write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x44, 0x44]))
read_data_from_module(ser)

# モジュールをリセット(コマンドコード:0x04)
print("resetting...")
write_data_to_module(ser, bytes([0x55, 0x07, 0x00, 0x04, 0x00, 0xFF, 0xFF, 0x00, 0x04]))
time.sleep(5)
read_data_from_module(ser)

# エンドポイントの設定をする
set_zcl_port_configuration(ser)

# モジュールをリセット(コマンドコード:0x04)
print("resetting...")
write_data_to_module(ser, bytes([0x55, 0x07, 0x00, 0x04, 0x00, 0xFF, 0xFF, 0x00, 0x04]))
time.sleep(5)
read_data_from_module(ser)

while True:
    # シリアルポートの読み込みバッファをクリア
    ser.reset_input_buffer()

    # 現在のノード情報を取得するコマンドの実行(コマンドコード:0x00)
    write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x00, 0x00]))
    recv_data = read_data_from_module(ser)

    # ノード情報からネットワークステータスとデバイスタイプ、MACアドレスを取得
    network_status = recv_data[4]
    device_type = recv_data[5]
    mac_addr = struct.unpack('<Q', recv_data[6:14])[0]

    print("[Node Status]")
    print(" network status : 0x%02X" % network_status)
    print(" device type    : 0x%02X" % device_type)
    print(" MAC address    : 0x%016X" % mac_addr)

    if network_status == 0x00:
        # 接続が成功したら情報を表示して終了する
        channel = recv_data[14]
        pan_id = struct.unpack('<H', recv_data[15:17])[0]
        short_addr = struct.unpack('<H', recv_data[17:19])[0]
        ext_pan_id = struct.unpack('<Q', recv_data[19:27])[0]
        net_key = recv_data[27:44]
        print(" channel        : 0x%02X (%d)" % (channel, channel))
        print(" PAN ID         : 0x%04X" % pan_id)
        print(" short address  : 0x%04X" % short_addr)
        print(" extended PAN ID: 0x%016X" % ext_pan_id)
        print(" network key    : ", end='')
        for b in net_key:
            print("%02X" % b, end=' ')
        print("")
        break
    else:
        # ネットワークに参加(コマンドコード:0x02)
        write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x02, 0x02]))
        print("connecting network...")
        time.sleep(10)

print("Completed.")
