"""
Z240-MP4をエンドデバイス(On/Offライト)として動作させる

実行方法:
    python run.py COM1
"""

import serial
import struct
import sys


# シリアルポートの設定
# コマンドライン引数でポート名(例:COM1,/dev/ttyUSB0)を指定する
PORT = sys.argv[1]
BAUDRATE = 115200
TIMEOUT = 1.0
READ_SIZE = 255

# スリープ型エンドデバイスとする場合はTrueを指定する
IS_SLEEPY_DEVICE = False


def write_data_to_module(ser: serial.Serial, data: bytes) -> None:
    # スリープ型エンドデバイスの場合、ウェイクアップのために先頭に0x00を6バイト書き込む必要がある
    if IS_SLEEPY_DEVICE:
        ser.write(bytes([0x00, 0x00, 0x00, 0x00, 0x00, 0x00]) + data)
    else:
        ser.write(data)
    print("WRITE:", ' '.join('%02X' % b for b in data))


def read_data_from_module(ser: serial.Serial) -> bytes:
    ret = ser.read(size=READ_SIZE)
    if len(ret) > 0:
        print("READ: ", ' '.join('%02X' % b for b in ret))
    return ret


def calc_xor8(data: bytes) -> int:
    # チェックサムを計算
    xor = 0
    for d in data:
        xor = xor ^ d

    return xor


def read_attribute(ser: serial.Serial, port_idx: int, direction: int, cluster_id: int,
                   manufacturer_code: int, attr_id: int) -> int:
    # 現在のOnOff属性値を取得する(コマンドコード:0x43)

    # コマンドデータを作成
    cmd_data = bytearray([0x00, 0x43, 0x00, port_idx, direction])
    cmd_data += struct.pack('<H', cluster_id)
    cmd_data += struct.pack('<H', manufacturer_code)
    cmd_data += struct.pack('<H', attr_id)
    checksum = calc_xor8(cmd_data)
    cmd_data += bytes([checksum])
    frame_data = bytes([0x55, len(cmd_data)]) + cmd_data

   # コマンドを実行
    write_data_to_module(ser, frame_data)
    recv_data = read_data_from_module(ser)

    if len(recv_data) < 7:
        return None
    # 結果から属性値のデータを取り出す
    status = recv_data[4]
    data_type = recv_data[5]
    data_value = recv_data[6]

    return data_value


def write_attribute(ser: serial.Serial, port_idx: int, direction: int, cluster_id: int,
                    manufacturer_code: int, attr_id: int, onoff: int) -> None:
    # OnOff属性値を設定する(コマンドコード:0x43)

    # コマンドデータを作成
    cmd_data = bytearray([0x00, 0x43, 0x01, port_idx, direction])
    cmd_data += struct.pack('<H', cluster_id)
    cmd_data += struct.pack('<H', manufacturer_code)
    cmd_data += struct.pack('<H', attr_id)
    cmd_data += bytes([onoff])
    checksum = calc_xor8(cmd_data)
    cmd_data += bytes([checksum])
    frame_data = bytes([0x55, len(cmd_data)]) + cmd_data

   # コマンドを実行
    write_data_to_module(ser, frame_data)
    read_data_from_module(ser)


def parse_message(data: bytes) -> None:
    # 受信したメッセージを解析する

    # 先頭1バイトは必ず開始コード0x55
    if data[0] != 0x55:
        return

    # フレーム長を確認
    frame_length = data[1]
    if frame_length != len(data[2:]):
        print("Error! Invalid frame length.")
        return

    # コマンドタイプとコマンドコードを取得
    cmd_type = data[2]
    cmd_code = data[3]

    # 受信メッセージがZCL制御コマンドの場合、後に続くパラメータを読み込む
    if cmd_type == 0x82 and cmd_code == 0x0F:
        payload = data[4:15]
        mode, src_addr, src_port, frame_num, direction, cluster_id, manufacturer_code, rssi = \
            struct.unpack('<BHBBBHHB', payload)
        port_idx = mode & 0x0f
        ext_data = data[15:]
        zcl_command_id = ext_data[0]

        print("receive mode : 0x%02X (port index: %d)" % (mode, port_idx))
        print("src address  : 0x%04X" % src_addr)
        print("src port     :", src_port)
        print("frame number :", frame_num)
        print("direction    : 0x%02X" % direction)
        print("cluster id   : 0x%04X" % cluster_id)
        print("manufacturer : 0x%04X" % manufacturer_code)
        print("RSSI         : 0x%02X (%d dBm)" % (rssi, rssi - 256))
        print("command id   : 0x%02X" % zcl_command_id)

        # OnOff属性のAttribute Identifier=0x0000
        attr_id = 0x0000

        # Command ID=0x00(Off)を受信
        if zcl_command_id == 0x00:
            print("Light [Off]")
            write_attribute(ser, port_idx, direction, cluster_id, manufacturer_code, attr_id, 0)
        # Command ID=0x01(On)を受信
        elif zcl_command_id == 0x01:
            print("Light [On]")
            write_attribute(ser, port_idx, direction, cluster_id, manufacturer_code, attr_id, 1)
        # Command ID=0x02(Toggle)を受信
        elif zcl_command_id == 0x02:
            print("Light [Toggle]")
            current_onoff = read_attribute(ser, port_idx, direction,
                                           cluster_id, manufacturer_code, attr_id)
            if current_onoff is not None:
                write_attribute(ser, port_idx, direction, cluster_id,
                                manufacturer_code, attr_id, (not current_onoff))


# シリアルポートを開く
ser = serial.Serial(port=PORT, baudrate=BAUDRATE, timeout=TIMEOUT)

print("Starting the ON/OFF light device.")

# シリアルポートの読み込みタイムアウトを0.01秒に設定
ser.timeout = 0.01
#write_attribute(ser, 0, 0, 0x0006, 0x0000, 0x0000, 1)
while True:
    # シリアルポートからのメッセージを受信待機
    recv_data = read_data_from_module(ser)
    if len(recv_data) > 0:
        parse_message(recv_data)
