"""
Z240-MP4, Z240-CP13コーディネータ
デバイス管理用クラスの定義
"""

from copy import deepcopy
from time import localtime, strftime,  time
from typing import Any


class Attribute:
    def __init__(self, id: int, value: Any = None) -> None:
        self.id = id
        self.value = value
        self.last_updated = None

    def set_value(self, value: Any) -> None:
        self.value = value
        self.last_updated = time()

    def get_latest_time(self) -> str | None:
        if self.last_updated is not None:
            return strftime("%Y-%m-%d %H:%M:%S", localtime(self.last_updated))
        else:
            return None

    def to_dict(self) -> dict:
        return {"id": f'0x{self.id:04X}', "value": self.value,
                "last_updated": self.get_latest_time()}


class Cluster:
    def __init__(self, id: int) -> None:
        self.id = id
        self.manufacturer_code = 0x0000
        self.attribute_list: list[Attribute] = []

    def add_attribute(self, attribute: Attribute) -> None:
        self.attribute_list.append(attribute)

    def set_attribute_list(self, attribute_list: list[Attribute]) -> None:
        self.attribute_list = deepcopy(attribute_list)

    def get_attribute_by_id(self, attribute_id: int) -> Attribute | None:
        for attr in self.attribute_list:
            if attr.id == attribute_id:
                return attr
        return None

    def to_dict(self) -> dict:
        return {"id": f'0x{self.id:04X}',
                "attributes": [attr.to_dict() for attr in self.attribute_list]}


class EndPoint:
    def __init__(self, port: int, profile_id: int,  device_id: int,
                 in_cluster_list: list[Cluster], out_cluster_list: list[Cluster]) -> None:
        self.port = port
        self.profile_id = profile_id
        self.device_id = device_id
        self.in_cluster_list = in_cluster_list
        self.out_cluster_list = out_cluster_list

    def set_in_cluster_list(self, in_cluster_list: list[Cluster]) -> None:
        self.in_cluster_list = deepcopy(in_cluster_list)

    def set_out_cluster_list(self, out_cluster_list: list[Cluster]) -> None:
        self.out_cluster_list = deepcopy(out_cluster_list)

    def get_cluster_by_id(self, cluster_id: int) -> Cluster | None:
        for cl in self.in_cluster_list:
            if cl.id == cluster_id:
                return cl
        return None

    def to_dict(self) -> dict:
        return {"port": self.port,
                "profile_id": f'0x{self.profile_id:04X}',
                "device_id": f'0x{self.device_id:04X}',
                "in_clusters": [cl.to_dict() for cl in self.in_cluster_list],
                "out_clusters": [cl.to_dict() for cl in self.out_cluster_list]
                }


class Device:
    def __init__(self, mac_addr: int, short_addr: int) -> None:
        self.mac_addr = mac_addr
        self.short_addr = short_addr
        self.endpoint_list: list[EndPoint] = []

    def update(self, short_addr: int) -> None:
        self.short_addr = short_addr
        self.endpoint_list = []

    def add_endpoint(self, endpoint: EndPoint) -> None:
        for ep in self.endpoint_list:
            if ep.port == endpoint.port:
                self.endpoint_list.remove(ep)
        self.endpoint_list.append(deepcopy(endpoint))

    def get_endpoint_by_port_number(self, port: int) -> EndPoint | None:
        for ep in self.endpoint_list:
            if ep.port == port:
                return ep
        return None

    def to_dict(self) -> dict:
        return {"mac_addr": f'0x{self.mac_addr:016X}',
                "short_addr": f'0x{self.short_addr:04X}',
                "endpoints": [ep.to_dict() for ep in self.endpoint_list]}


class DeviceManager:
    def __init__(self) -> None:
        self.device_list: list[Device] = []

    def add_device(self, device: Device) -> bool:
        if self.has_device(device.mac_addr):
            return False
        print("[MESSAGE] New device has joined. addr=0x%04X MAC=0x%016X"
              % (device.short_addr, device.mac_addr))
        self.device_list.append(deepcopy(device))
        return True

    def remove_device_by_short_addr(self, short_addr: int) -> bool:
        for device in self.device_list:
            if device.short_addr == short_addr:
                print("[MESSAGE] Device has left. addr=0x%04X MAC=0x%016X"
                      % (device.short_addr, device.mac_addr))
                self.device_list.remove(device)
                return True
        return False

    def remove_device_by_mac_addr(self, mac_addr: int) -> bool:
        for device in self.device_list:
            if device.mac_addr == mac_addr:
                print("[MESSAGE] Device has left. addr=0x%04X MAC=0x%016X"
                      % (device.short_addr, device.mac_addr))
                self.device_list.remove(device)
                return True
        return False

    def update_device(self, mac_addr: int, short_addr: int) -> None:
        for device in self.device_list:
            if device.mac_addr == mac_addr:
                device.update(short_addr)

    def has_device(self, mac_addr: int) -> bool:
        for device in self.device_list:
            if device.mac_addr == mac_addr:
                return True
        return False

    def get_device_by_short_addr(self, short_addr: int) -> Device | None:
        for device in self.device_list:
            if device.short_addr == short_addr:
                return device
        return None

    def to_dict(self) -> dict:
        return {"devices": [dev.to_dict() for dev in self.device_list]}
