"""
Z240-MP4, Z240-CP13をコーディネータとして設定する
ネットワークへのデバイス参加許可を開始する

実行方法:
    python config.py COM1
"""

import serial
import struct
import sys
import time

# モデル種別の定義
MODEL_Z240_MP4 = 1
MODEL_Z240_CP13 = 2

# シリアルポートの設定
# コマンドライン引数でポート名(例:COM1,/dev/ttyUSB0)を指定する
PORT = sys.argv[1]
BAUDRATE = 115200
TIMEOUT = 0.5
READ_SIZE = 255

# 使用するモジュールのモデルを指定する
device_model = MODEL_Z240_MP4

def write_data_to_module(ser: serial.Serial, data: bytes) -> None:
    ser.write(data)
    print("WRITE:", ' '.join('%02X' % b for b in data))

def read_data_from_module(ser: serial.Serial) -> bytes:
    ret = ser.read(size=READ_SIZE)
    print("READ: ", ' '.join('%02X' % b for b in ret))
    return ret


# シリアルポートを開く
ser = serial.Serial(port=PORT, baudrate=BAUDRATE, timeout=TIMEOUT)

# モジュールをコーディネータに設定(コマンドコード:0x05)
print("set device type to Coordinator")
write_data_to_module(ser, bytes([0x55, 0x04, 0x00, 0x05, 0x00, 0x05]))
read_data_from_module(ser)

# モジュールをリセット(コマンドコード:0x04)
print("resetting...")
write_data_to_module(ser, bytes([0x55, 0x07, 0x00, 0x04, 0x00, 0xFF, 0xFF, 0x00, 0x04]))
time.sleep(3)
read_data_from_module(ser)

# 送信出力を最大に設定
if device_model == MODEL_Z240_MP4:
    # 送信出力を4dBmに設定する(コマンドコード:0x0D)
    write_data_to_module(ser, bytes([0x55, 0x05, 0x00, 0x0D, 0x01, 0x04, 0x08]))
    read_data_from_module(ser)

elif device_model == MODEL_Z240_CP13:
    # 送信出力を13dBmに設定する(コマンドコード:0x0D)
    write_data_to_module(ser, bytes([0x55, 0x05, 0x00, 0x0D, 0x01, 0x0D, 0x01]))
    read_data_from_module(ser)

# コーディネータがネットワークを作成、開始する(コマンドコード:0x02)
print("starting network...")
write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x02, 0x02]))
time.sleep(10)
read_data_from_module(ser)

# シリアルポートの読み込みバッファをクリア
ser.reset_input_buffer()

# 現在のノード情報を取得するコマンドの実行(コマンドコード:0x00)
write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x00, 0x00]))
recv_data = read_data_from_module(ser)

# ノード情報からネットワークステータスとデバイスタイプ、MACアドレスを取得
network_status = recv_data[4]
device_type = recv_data[5]
mac_addr = struct.unpack('<Q', recv_data[6:14])[0]

print("[Node Status]")
print(" network status : 0x%02X" % network_status)
print(" device type    : 0x%02X" % device_type)
print(" MAC address    : 0x%016X" % mac_addr)

# ネットワーク作成済みの場合、ネットワーク情報を表示
if network_status == 0x00:
    channel = recv_data[14]
    pan_id = struct.unpack('<H', recv_data[15:17])[0]
    short_addr = struct.unpack('<H', recv_data[17:19])[0]
    ext_pan_id = struct.unpack('<Q', recv_data[19:27])[0]
    net_key = recv_data[27:44]
    print(" channel        : 0x%02X (%d)" % (channel, channel))
    print(" PAN ID         : 0x%04X" % pan_id)
    print(" short address  : 0x%04X" % short_addr)
    print(" extended PAN ID: 0x%016X" % ext_pan_id)
    print(" network key    : ", end='')
    for b in net_key:
        print("%02X" % b, end=' ')
