"""
Z240-MP4, Z240-CP13をコーディネータとして使用する
エンドデバイスから送信された文字列データを受信する

実行方法:
    python coodinator.py COM1
"""

import serial
import struct
import sys
import time

# シリアルポートの設定
# コマンドライン引数でポート名(例:COM1,/dev/ttyUSB0)を指定する
PORT = sys.argv[1]
BAUDRATE = 115200
TIMEOUT = 1.0
READ_SIZE = 255


def write_data_to_module(ser: serial.Serial, data: bytes, is_sleepy: bool = False) -> None:
    # スリープ型エンドデバイスの場合、ウェイクアップのために先頭に0x00を6バイト書き込む必要がある
    if is_sleepy:
        ser.write(bytes([0x00, 0x00, 0x00, 0x00, 0x00, 0x00]) + data)
    else:
        ser.write(data)


def read_data_from_module(ser: serial.Serial) -> bytes:
    ret = ser.read(size=READ_SIZE)

    return ret


# シリアルポートを開く
ser = serial.Serial(port=PORT, baudrate=BAUDRATE, timeout=TIMEOUT)

# 1バイトの書き込みの応答がタイムアウトの場合、スリープ型エンドデバイスと判断する
write_data_to_module(ser, bytes([0x00]), False)
recv_data = read_data_from_module(ser)

if len(recv_data) == 0:
    is_sleepy = True
else:
    is_sleepy = False

# 現在のシリアルポート通信のモードを確認するために1バイトの書き込みを行う
write_data_to_module(ser, bytes([0x00]), is_sleepy)
recv_data = read_data_from_module(ser)

# 現在のシリアルポート通信が透過伝送モードの場合、一度HEXコマンドモードに変更する
if b'OK' in recv_data or b'ERRO' in recv_data:
    write_data_to_module(ser, b'+++', False)
    recv_data = read_data_from_module(ser)


while True:
    # シリアルポートの読み込みバッファをクリア
    ser.reset_input_buffer()

    # 現在のノード情報を取得するコマンドの実行(コマンドコード:0x00)
    write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x00, 0x00]), is_sleepy)
    recv_data = read_data_from_module(ser)
    if len(recv_data) < 6:
        print("Message error!")
        continue

    # ノード情報からネットワークステータスとデバイスタイプを取得
    network_status = recv_data[4]
    device_type = recv_data[5]
    print("network status: 0x%02X" % network_status)
    print("device type: 0x%02X" % device_type)
    break

# 既にネットワーク作成済みの場合、ネットワーク情報を表示
if network_status == 0x00:
    short_addr = struct.unpack('<H', recv_data[17:19])[0]
    pan_id = struct.unpack('<H', recv_data[15:17])[0]
    channel = recv_data[14]
    print("short address: 0x%04X" % short_addr)
    print("PAN ID: 0x%04X" % pan_id)
    print("channel: 0x%02X" % channel)

# デバイスタイプがコーディネータでない場合
if network_status != 0x00 and device_type != 0x00:
    # コーディネータに設定(コマンドコード:0x05)
    write_data_to_module(ser, bytes([0x55, 0x04, 0x00, 0x05, 0x00, 0x05]))
    print("change device type")
    # モジュールをリセット(コマンドコード:0x04)
    write_data_to_module(ser, bytes([0x55, 0x07, 0x00, 0x04, 0x00, 0xFF, 0xFF, 0x00, 0x04]))
    time.sleep(3)
    is_sleepy = False

# コーディネータがネットワークを作成、開始する(コマンドコード:0x02)
print("starting network...")
write_data_to_module(ser, bytes([0x55, 0x03, 0x00, 0x02, 0x02]))
time.sleep(10)

# 送信出力を4dBmに設定する(コマンドコード:0x0D)
write_data_to_module(ser, bytes([0x55, 0x05, 0x00, 0x0D, 0x01, 0x04, 0x08]))
read_data_from_module(ser)

# シリアルポート通信を透過伝送モードに設定する(コマンドコード:0x11)
write_data_to_module(ser, bytes([0x55, 0x07, 0x00, 0x11, 0x00, 0x03, 0x00, 0x01, 0x13]))
read_data_from_module(ser)

# シリアルポートの読み込みバッファをクリア
ser.reset_input_buffer()

# シリアルポートの読み込みバッファを0.1秒ごとにチェック、受信データがあれば表示する
print("waiting to receive a message")
ser.timeout = 0.1
while True:
    try:
        buffer = read_data_from_module(ser)
        if len(buffer) > 0:
            print(f"[{time.time():.3f}] received:", buffer.decode())
    except UnicodeDecodeError:
        pass
