"""
Z240-CP13をルータとして使用する

実行方法:
    python router.py COM1
"""

import serial
import sys
import time

# シリアルポートの設定
# コマンドライン引数でポート名(例:COM1,/dev/ttyUSB0)を指定する
PORT = sys.argv[1]
BAUDRATE = 115200
TIMEOUT = 1.0
READ_SIZE = 255


def write_text_to_module(ser: serial.Serial, text: str, is_sleepy: bool = False) -> None:
    # スリープ型のエンドデバイスの場合、6バイト分の0x00を書き込むデータの前に追加する
    if is_sleepy:
        ser.write(bytes([0x00, 0x00, 0x00, 0x00, 0x00, 0x00]) + text.encode())
    else:
        ser.write(text.encode())


def read_text_from_module(ser: serial.Serial) -> str:
    try:
        data = ser.read(size=READ_SIZE)
        ret = data.decode().rstrip('\r\n')
        return ret
    except UnicodeDecodeError:
        return ''


# シリアルポートを開く
ser = serial.Serial(port=PORT, baudrate=BAUDRATE, timeout=TIMEOUT)

# 1バイトの書き込みの応答がタイムアウトの場合、スリープ型エンドデバイスと判断する
write_text_to_module(ser, 'z', False)
recv_data = read_text_from_module(ser)

if len(recv_data) == 0:
    is_sleepy = True
else:
    is_sleepy = False

# 現在のシリアルポート通信のモードを確認するために1バイトの書き込みを行う
write_text_to_module(ser, 'z', is_sleepy)
recv_data = read_text_from_module(ser)

# 現在のシリアルポート通信が透過伝送モードの場合、一度ATコマンドモードに変更する
if 'OK' in recv_data or 'ERRO' in recv_data:
    write_text_to_module(ser, '+AT', is_sleepy)
    read_text_from_module(ser)

while True:
    # 現在のノード情報を取得するコマンドの実行(コマンド:AT+INFO)
    write_text_to_module(ser, 'AT+INFO?', is_sleepy)
    recv_data = read_text_from_module(ser)
    if not 'VERSION' in recv_data:
        print("Message error!")
        continue
    print("device infomation:", recv_data)

    # ノード情報からネットワークステータスを取得
    for line in recv_data.splitlines():
        # ネットワーク接続済み
        if 'PANID=' in line:
            is_networking = True
            break
        # ネットワーク未接続
        if 'NO NET' in line:
            is_networking = False
            break

    if is_networking is True:
        break

    # デバイスタイプの確認(コマンド:AT+DEVTYPE)
    write_text_to_module(ser, 'AT+DEVTYPE?', is_sleepy)
    recv_data = read_text_from_module(ser)
    if not 'DEVTYPE' in recv_data:
        print("Message error!")
        continue
    device_type = recv_data.split('=')[1]
    print("device type:", device_type)

    # デバイスタイプがルータでない場合
    if device_type != 'SLEEPY_END_DEVICE':
        # ルータに設定(コマンド:AT+DEVTYPE)
        write_text_to_module(ser, 'AT+DEVTYPE=1', is_sleepy)
        print("change device type")
        # モジュールをリセット(コマンド:AT+RESET)
        write_text_to_module(ser, 'AT+RESET', is_sleepy)
        print("resetting...")
        time.sleep(3)
        is_sleepy = False

    # ネットワークに参加(コマンド:AT+JOIN)
    write_text_to_module(ser, 'AT+JOIN', False)
    print("connecting network...")
    time.sleep(10)
    ser.reset_input_buffer()
