"""
Z240-CP13のネットワーク設定をリセットする
シリアルポート通信をATコマンドモードに変更する

実行方法:
    python reset.py COM1
"""

import serial
import sys

# シリアルポートの設定
# コマンドライン引数でポート名(例:COM1,/dev/ttyUSB0)を指定する
PORT = sys.argv[1]
READ_SIZE = 255
TIMEOUT = 1.0


def write_command(ser: serial.Serial, data: bytes, is_sleepy: bool = False) -> bytes:
    # スリープ型エンドデバイスの場合、ウェイクアップのために先頭に0x00を6バイト書き込む必要がある
    if is_sleepy:
        ser.write(bytes([0x00, 0x00, 0x00, 0x00, 0x00, 0x00]) + data)
    else:
        ser.write(data)

    ret = ser.read(size=READ_SIZE)

    return ret


def reset_module(ser: serial.Serial) -> bool:
    # 1バイトの書き込みの応答がタイムアウトの場合、スリープ型エンドデバイスと判断する
    recv_data = write_command(ser, bytes([0x00]), False)

    if len(recv_data) == 0:
        is_sleepy = True
    else:
        is_sleepy = False

    # 現在のシリアルポート通信のモードを確認するために1バイトの書き込みを行う
    recv_data = write_command(ser, bytes([0x00]), is_sleepy)

    # 透過伝送モードの場合、HEXコマンドモードに戻す
    if b'OK' in recv_data or b'ERRO' in recv_data:
        write_command(ser, b'+++', False)
        write_command(ser, b'+++', True)
    # ATコマンドモードの場合、HEXコマンドモードに戻す
    if b'WRONG' in recv_data:
        write_command(ser, b'AT+EXIT', is_sleepy)

    # シリアルポートの読み込みバッファをクリア
    ser.reset_input_buffer()

    # 現在のノード情報を取得するコマンドの実行(コマンドコード:0x00)
    recv_data = write_command(ser, bytes([0x55, 0x03, 0x00, 0x00, 0x00]), is_sleepy)

    if len(recv_data) == 44 and recv_data[0:4] == bytes([0x55, 0x2A, 0x00, 0x00]):
        # ネットワークに参加している場合、ネットワークから離脱してリセットする

        # ノード情報からPAN IDとチャンネル番号を取り出す
        pan_id = recv_data[15:17]
        channel = recv_data[14]
        print(f"PAN ID: 0x{pan_id[1]:02X}{pan_id[0]:02X}")
        print(f"channel: 0x{channel:02X}")

        # リセットコマンドのバイト列を作成(コマンドコード:0x04)
        cmd_data = bytearray(
            [0x00, 0x04, 0x02, pan_id[0], pan_id[1], channel])

        # 末尾にチェックサムを追加
        checksum = 0
        for d in cmd_data:
            checksum ^= d
        cmd_data.append(checksum)

        # 先頭に開始コード0x55とコマンドバイト列のサイズを追加
        cmd_data = bytearray([0x55, len(cmd_data)]) + cmd_data

        # モジュールにコマンドデータを書き込み
        write_command(ser, cmd_data, is_sleepy)
        return True
    elif len(recv_data) == 15 and recv_data[0:4] == bytes([0x55, 0x0D, 0x00, 0x00]):
        # ネットワークに参加していない場合、モジュールを再起動する(コマンドコード:0x04)
        write_command(ser, bytes(
            [0x55, 0x07, 0x00, 0x04, 0x00, 0xFF, 0xFF, 0x00, 0x04]), is_sleepy)
        return True

    return False


# ボーレート設定を変えてリセットを順番に試す
for b in [115200, 58600, 38400, 19200, 9600]:
    print("baudrate:", b)
    with serial.Serial(port=PORT, baudrate=b, timeout=TIMEOUT) as ser:
        if reset_module(ser) is True:
            print("Reset OK")
            # ATコマンドモードに切り替え(コマンドコード:0x16)
            write_command(ser, bytes([0x55, 0x03, 0x00, 0x16, 0x16]), True)
            break

print("done")
